// SPDX-License-Identifier: GPL-2.0+
/*
 * efi_selftest_devicepath
 *
 * Copyright (c) 2020 Heinrich Schuchardt <xypron.glpk@gmx.de>
 *
 * Test the EFI_TCG2_PROTOCOL
 */

#include <efi_selftest.h>
#include <efi_tcg2.h>
/*
 * Include containing the miniapp.efi application.
 * Note that tcg2 selftest measures the PE/COFF image,
 * so we must have the pre-build efi application for
 * each architecture.
 */
#if defined(__arm__)
#include "efi_miniapp_tcg2_arm.h"
#elif defined(__aarch64__)
#include "efi_miniapp_tcg2_arm64.h"
#elif defined(__i386__)
#include "efi_miniapp_tcg2_ia32.h"
#elif defined(__x86_64__)
#include "efi_miniapp_tcg2_x86_64.h"
#elif defined(__riscv) && (__riscv_xlen == 32)
#include "efi_miniapp_tcg2_riscv32.h"
#elif defined(__riscv) && (__riscv_xlen == 64)
#include "efi_miniapp_tcg2_riscv64.h"
#endif

#include <linux/unaligned/be_byteshift.h>
#include <linux/unaligned/le_byteshift.h>
#include <mapmem.h>
#include <smbios.h>
#include <tables_csum.h>

static struct efi_boot_services *boottime;
static const efi_guid_t guid_tcg2 = EFI_TCG2_PROTOCOL_GUID;

/* Block size of compressed disk image */
#define COMPRESSED_DISK_IMAGE_BLOCK_SIZE 8

static efi_handle_t image_handle;
/* Decompressed file image */
static u8 *image;

/* One 8 byte block of the compressed disk image */
struct line {
	size_t addr;
	char *line;
};

/* Compressed file image */
struct compressed_file_image {
	size_t length;
	struct line lines[];
};

static struct compressed_file_image img = EFI_ST_DISK_IMG;

static struct efi_tcg2_event *efi_tcg2_event;

static struct efi_runtime_services *runtime;
#define BOOT_NAME_1000 u"Boot1000"
#define BOOT_NAME_1001 u"Boot1001"
#define BOOT_NAME_1002 u"Boot1002"

#define DEFAULT_ATTR (EFI_VARIABLE_NON_VOLATILE | \
		      EFI_VARIABLE_BOOTSERVICE_ACCESS | \
		      EFI_VARIABLE_RUNTIME_ACCESS)

/* "efidebug boot add -b 1000 test1000 virtio 0:1 /EFI/debian/grubaa64.efi" */
static const u8 boot_1000[] = {
0x01, 0x00, 0x00, 0x00, 0x8d, 0x00, 0x74, 0x00, 0x65, 0x00, 0x73, 0x00,
0x74, 0x00, 0x30, 0x00, 0x00, 0x00, 0x01, 0x04, 0x14, 0x00, 0xb9, 0x73,
0x1d, 0xe6, 0x84, 0xa3, 0xcc, 0x4a, 0xae, 0xab, 0x82, 0xe8, 0x28, 0xf3,
0x62, 0x8b, 0x01, 0x04, 0x15, 0x00, 0x92, 0x37, 0x29, 0x63, 0xf5, 0xad,
0x25, 0x93, 0xb9, 0x9f, 0x4e, 0x0e, 0x45, 0x5c, 0x1b, 0x1e, 0x00, 0x04,
0x01, 0x2a, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x57,
0x5a, 0x47, 0xc2, 0x35, 0x27, 0x44, 0x47, 0x9f, 0x01, 0x67, 0xfe, 0xfa,
0x1d, 0x06, 0xae, 0x02, 0x02, 0x04, 0x04, 0x36, 0x00, 0x5c, 0x00, 0x45,
0x00, 0x46, 0x00, 0x49, 0x00, 0x5c, 0x00, 0x64, 0x00, 0x65, 0x00, 0x62,
0x00, 0x69, 0x00, 0x61, 0x00, 0x6e, 0x00, 0x5c, 0x00, 0x67, 0x00, 0x72,
0x00, 0x75, 0x00, 0x62, 0x00, 0x61, 0x00, 0x61, 0x00, 0x36, 0x00, 0x34,
0x00, 0x2e, 0x00, 0x65, 0x00, 0x66, 0x00, 0x69, 0x00, 0x00, 0x00, 0x7f,
0xff, 0x04, 0x00 };

/* "efidebug boot add -b 1001 test1001 virtio 0:1 /EFI/debian/grubaa64.efi" */
static const u8 boot_1001[] = {
0x01, 0x00, 0x00, 0x00, 0x8d, 0x00, 0x74, 0x00, 0x65, 0x00, 0x73, 0x00,
0x74, 0x00, 0x31, 0x00, 0x00, 0x00, 0x01, 0x04, 0x14, 0x00, 0xb9, 0x73,
0x1d, 0xe6, 0x84, 0xa3, 0xcc, 0x4a, 0xae, 0xab, 0x82, 0xe8, 0x28, 0xf3,
0x62, 0x8b, 0x01, 0x04, 0x15, 0x00, 0x92, 0x37, 0x29, 0x63, 0xf5, 0xad,
0x25, 0x93, 0xb9, 0x9f, 0x4e, 0x0e, 0x45, 0x5c, 0x1b, 0x1e, 0x00, 0x04,
0x01, 0x2a, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x57,
0x5a, 0x47, 0xc2, 0x35, 0x27, 0x44, 0x47, 0x9f, 0x01, 0x67, 0xfe, 0xfa,
0x1d, 0x06, 0xae, 0x02, 0x02, 0x04, 0x04, 0x36, 0x00, 0x5c, 0x00, 0x45,
0x00, 0x46, 0x00, 0x49, 0x00, 0x5c, 0x00, 0x64, 0x00, 0x65, 0x00, 0x62,
0x00, 0x69, 0x00, 0x61, 0x00, 0x6e, 0x00, 0x5c, 0x00, 0x67, 0x00, 0x72,
0x00, 0x75, 0x00, 0x62, 0x00, 0x61, 0x00, 0x61, 0x00, 0x36, 0x00, 0x34,
0x00, 0x2e, 0x00, 0x65, 0x00, 0x66, 0x00, 0x69, 0x00, 0x00, 0x00, 0x7f,
0xff, 0x04, 0x00 };

/* "efidebug boot add -b 1002 test1002 virtio 0:1 /EFI/debian/grubaa64.efi" */
static const u8 boot_1002[] = {
0x01, 0x00, 0x00, 0x00, 0x8d, 0x00, 0x74, 0x00, 0x65, 0x00, 0x73, 0x00,
0x74, 0x00, 0x32, 0x00, 0x00, 0x00, 0x01, 0x04, 0x14, 0x00, 0xb9, 0x73,
0x1d, 0xe6, 0x84, 0xa3, 0xcc, 0x4a, 0xae, 0xab, 0x82, 0xe8, 0x28, 0xf3,
0x62, 0x8b, 0x01, 0x04, 0x15, 0x00, 0x92, 0x37, 0x29, 0x63, 0xf5, 0xad,
0x25, 0x93, 0xb9, 0x9f, 0x4e, 0x0e, 0x45, 0x5c, 0x1b, 0x1e, 0x00, 0x04,
0x01, 0x2a, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x57,
0x5a, 0x47, 0xc2, 0x35, 0x27, 0x44, 0x47, 0x9f, 0x01, 0x67, 0xfe, 0xfa,
0x1d, 0x06, 0xae, 0x02, 0x02, 0x04, 0x04, 0x36, 0x00, 0x5c, 0x00, 0x45,
0x00, 0x46, 0x00, 0x49, 0x00, 0x5c, 0x00, 0x64, 0x00, 0x65, 0x00, 0x62,
0x00, 0x69, 0x00, 0x61, 0x00, 0x6e, 0x00, 0x5c, 0x00, 0x67, 0x00, 0x72,
0x00, 0x75, 0x00, 0x62, 0x00, 0x61, 0x00, 0x61, 0x00, 0x36, 0x00, 0x34,
0x00, 0x2e, 0x00, 0x65, 0x00, 0x66, 0x00, 0x69, 0x00, 0x00, 0x00, 0x7f,
0xff, 0x04, 0x00};

/* "efidebug boot order 1002 1000 1001" */
static u8 boot_order[] = {0x02, 0x10, 0x00, 0x10, 0x01, 0x10};

static void *orig_smbios_table;
static u64 dmi_addr = U32_MAX;
#define SMBIOS3_ENTRY_HEADER_SIZE 0x18
/* smbios table for the measurement test */
static u8 smbios3_table_test[] = {
0x5f, 0x53, 0x4d, 0x33, 0x5f, 0x00, 0x18, 0x03, 0x07, 0x00, 0x01, 0x00,
0x5c, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x18, 0x00, 0x00, 0x01, 0x02, 0x00, 0x00, 0x03, 0x00, 0x80, 0x08,
0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x15, 0x0a, 0xff, 0xff,
0x55, 0x2d, 0x42, 0x6f, 0x6f, 0x74, 0x00, 0x32, 0x30, 0x32, 0x31, 0x2e,
0x31, 0x30, 0x2d, 0x72, 0x63, 0x34, 0x2d, 0x30, 0x30, 0x30, 0x30, 0x35,
0x2d, 0x67, 0x37, 0x32, 0x37, 0x63, 0x33, 0x66, 0x33, 0x32, 0x35, 0x39,
0x2d, 0x64, 0x69, 0x72, 0x74, 0x79, 0x00, 0x31, 0x30, 0x2f, 0x30, 0x31,
0x2f, 0x32, 0x30, 0x32, 0x31, 0x00, 0x00, 0x01, 0x1b, 0x01, 0x00, 0x01,
0x02, 0x00, 0x03, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x55, 0x6e,
0x6b, 0x6e, 0x6f, 0x77, 0x6e, 0x00, 0x55, 0x6e, 0x6b, 0x6e, 0x6f, 0x77,
0x6e, 0x20, 0x50, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x74, 0x00, 0x31, 0x32,
0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x00, 0x00, 0x02, 0x0e, 0x02, 0x00,
0x01, 0x02, 0x00, 0x04, 0x03, 0x01, 0x01, 0x01, 0x00, 0x0a, 0x55, 0x6e,
0x6b, 0x6e, 0x6f, 0x77, 0x6e, 0x00, 0x55, 0x6e, 0x6b, 0x6e, 0x6f, 0x77,
0x6e, 0x20, 0x50, 0x72, 0x6f, 0x64, 0x75, 0x63, 0x74, 0x00, 0x33, 0x33,
0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x00, 0x31, 0x32, 0x33, 0x34, 0x35,
0x36, 0x37, 0x38, 0x00, 0x00, 0x03, 0x15, 0x03, 0x00, 0x01, 0x03, 0x00,
0x02, 0x03, 0x03, 0x03, 0x03, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x55, 0x6e, 0x6b, 0x6e, 0x6f, 0x77, 0x6e, 0x00, 0x31, 0x32,
0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x00, 0x33, 0x33, 0x33, 0x33, 0x33,
0x33, 0x33, 0x33, 0x00, 0x00, 0x04, 0x30, 0x04, 0x00, 0x00, 0x03, 0x02,
0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x0c, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x06, 0xff, 0xff, 0xff, 0xff, 0xff,
0xff, 0x02, 0x03, 0x04, 0x04, 0x04, 0x08, 0x00, 0x00, 0x02, 0x00, 0x08,
0x00, 0x08, 0x00, 0x01, 0x00, 0x55, 0x6e, 0x6b, 0x6e, 0x6f, 0x77, 0x6e,
0x00, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x00, 0x33, 0x33,
0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x00, 0x35, 0x35, 0x35, 0x35, 0x35,
0x35, 0x35, 0x35, 0x00, 0x00, 0x20, 0x0b, 0x05, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0x04, 0x06, 0x00, 0x00, 0x00
};

#define IDX_ARRAY_SZ 3 /* support 24 PCRs */
#define TPM2_CMD_BUF_SIZE 64
/* TPM command is big endian */
#define __MSB(x) ((x) >> 8)
#define __LSB(x) ((x) & 0xFF)
#define tpm_u16(x) __MSB(x), __LSB(x)
#define tpm_u32(x) tpm_u16((x) >> 16), tpm_u16((x) & 0xFFFF)
#define TPM2_PCR_READ_HEADER_SIZE 30

static u8 (*pcrs)[TPM2_SHA256_DIGEST_SIZE];
static u8 expected_pcrs[EFI_TCG2_MAX_PCR_INDEX + 1][TPM2_SHA256_DIGEST_SIZE] = {
	{0x91, 0x21, 0x37, 0xc7, 0x1a, 0x49, 0x19, 0xc8,
	 0xf1, 0xfb, 0xa9, 0x84, 0x5c, 0x65, 0xa9, 0xdd,
	 0x7b, 0xb9, 0xfe, 0xa1, 0xcd, 0x64, 0x49, 0xdd,
	 0xed, 0xe2, 0x65, 0x82, 0xc5, 0x3e, 0xf4, 0xc4},

	{0x75, 0xb5, 0x91, 0x54, 0x12, 0xa8, 0xa4, 0x25,
	 0x73, 0x79, 0xa7, 0x47, 0xd9, 0x32, 0x54, 0x78,
	 0x9a, 0x80, 0x3f, 0xa8, 0x34, 0xfe, 0xd2, 0xae,
	 0x76, 0xd3, 0x16, 0x4a, 0xb2, 0x03, 0xac, 0xe6},

	{0x3d, 0x45, 0x8c, 0xfe, 0x55, 0xcc, 0x03, 0xea,
	 0x1f, 0x44, 0x3f, 0x15, 0x62, 0xbe, 0xec, 0x8d,
	 0xf5, 0x1c, 0x75, 0xe1, 0x4a, 0x9f, 0xcf, 0x9a,
	 0x72, 0x34, 0xa1, 0x3f, 0x19, 0x8e, 0x79, 0x69},

	{0x3d, 0x45, 0x8c, 0xfe, 0x55, 0xcc, 0x03, 0xea,
	 0x1f, 0x44, 0x3f, 0x15, 0x62, 0xbe, 0xec, 0x8d,
	 0xf5, 0x1c, 0x75, 0xe1, 0x4a, 0x9f, 0xcf, 0x9a,
	 0x72, 0x34, 0xa1, 0x3f, 0x19, 0x8e, 0x79, 0x69},

	/* PCR[4] is different per architecture */
	{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},

	{0x3d, 0x45, 0x8c, 0xfe, 0x55, 0xcc, 0x03, 0xea,
	 0x1f, 0x44, 0x3f, 0x15, 0x62, 0xbe, 0xec, 0x8d,
	 0xf5, 0x1c, 0x75, 0xe1, 0x4a, 0x9f, 0xcf, 0x9a,
	 0x72, 0x34, 0xa1, 0x3f, 0x19, 0x8e, 0x79, 0x69},

	/* PCR[6] is different per architecture */
	{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},

	{0x96, 0x74, 0xae, 0xcd, 0x3f, 0x40, 0xb4, 0xa9,
	 0x36, 0xae, 0x19, 0xc8, 0x84, 0x8a, 0xb9, 0x5a,
	 0x87, 0x99, 0xd8, 0x89, 0x7f, 0xfc, 0x40, 0x48,
	 0x05, 0x99, 0x65, 0x2e, 0x55, 0xd4, 0x93, 0x32},

	{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},

	{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},

	{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},

	{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},

	{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},

	{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},

	{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},

	{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},

	{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},

	{0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff},

	{0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff},

	{0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff},

	{0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff},

	{0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff},

	{0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff},

	{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}
};

/*
 * PCR[4] and PCR[6] have the PE/COFF image measurement,
 * this PCRs have different value in each architecture.
 */
#if defined(__arm__)
static u8 expected_pcrs_per_arch[][TPM2_SHA256_DIGEST_SIZE] = {
	/* PCR[4] */
	{0xcd, 0xa2, 0x06, 0xad, 0x83, 0x9b, 0x8f, 0x92,
	 0x01, 0xf9, 0xc8, 0x3d, 0xc9, 0x54, 0x66, 0xb3,
	 0x97, 0x35, 0x88, 0xe1, 0xea, 0xd3, 0x1a, 0xd6,
	 0x56, 0xee, 0x43, 0x1c, 0xdb, 0x4b, 0xf9, 0x1f},
	/* PCR[6] */
	{0x9c, 0xb8, 0x9d, 0x4a, 0xf6, 0x63, 0x95, 0xb0,
	 0x95, 0xfe, 0x44, 0x30, 0x0f, 0x3a, 0x0b, 0x7c,
	 0xef, 0xc7, 0xb6, 0x6a, 0x59, 0xae, 0xcb, 0xf6,
	 0xbd, 0x2d, 0xb5, 0xb9, 0xb4, 0x95, 0x7d, 0xaf}
};
#elif defined(__aarch64__)
static u8 expected_pcrs_per_arch[][TPM2_SHA256_DIGEST_SIZE] = {
	/* PCR[4] */
	{0x69, 0xdb, 0x01, 0x5e, 0x07, 0xed, 0x9c, 0xbb,
	 0x27, 0x65, 0xb1, 0xf0, 0x7b, 0x04, 0xbc, 0x31,
	 0xd1, 0xec, 0x00, 0xe4, 0xe1, 0x49, 0xdb, 0x1e,
	 0x8b, 0x2d, 0xa2, 0x26, 0xb5, 0x8d, 0x07, 0xe2},
	/* PCR[6] */
	{0x53, 0x1b, 0x27, 0xb2, 0x6f, 0x2d, 0xab, 0x9b,
	 0x6f, 0xbc, 0xd1, 0x8f, 0xc9, 0x14, 0x48, 0xe7,
	 0x6d, 0x1b, 0xfb, 0x1b, 0x53, 0xc5, 0x8e, 0xf4,
	 0x41, 0x50, 0x79, 0x24, 0x66, 0x57, 0x7b, 0xf8}
};
#elif defined(__i386__)
static u8 expected_pcrs_per_arch[][TPM2_SHA256_DIGEST_SIZE] = {
	/* PCR[4] */
	{0xec, 0x5e, 0xdb, 0x68, 0x13, 0x48, 0x36, 0x0a,
	 0x3a, 0xbc, 0x7b, 0x7b, 0xbc, 0x74, 0x7a, 0xa5,
	 0x55, 0xea, 0xb9, 0x09, 0x6a, 0x6e, 0xc3, 0x21,
	 0x51, 0x46, 0x22, 0xd2, 0x9d, 0xc9, 0xd5, 0x6a},
	/* PCR[6] */
	{0x26, 0x14, 0xe7, 0xde, 0x91, 0xd1, 0xf3, 0xde,
	 0x7a, 0xc2, 0x78, 0xaf, 0x4b, 0x2e, 0x05, 0x9d,
	 0x35, 0x17, 0xee, 0xcc, 0x0e, 0x77, 0x8d, 0x3f,
	 0x7e, 0x20, 0x75, 0xfa, 0xbc, 0xbc, 0x24, 0x3e}
};
#elif defined(__x86_64__)
static u8 expected_pcrs_per_arch[][TPM2_SHA256_DIGEST_SIZE] = {
	/* PCR[4] */
	{0x9a, 0x75, 0x99, 0x8b, 0x74, 0x45, 0xb6, 0x26,
	 0x50, 0xe0, 0xbb, 0xfa, 0x2a, 0xa6, 0x19, 0xec,
	 0x97, 0x12, 0x0c, 0xb5, 0xc8, 0x2a, 0xfe, 0xe5,
	 0x29, 0xc8, 0xd3, 0x98, 0xe9, 0xd1, 0x9d, 0xd5},
	/* PCR[6] */
	{0xa2, 0xa2, 0xd3, 0xa7, 0x84, 0xc2, 0x95, 0x2a,
	 0xab, 0x6f, 0xe7, 0xe8, 0x86, 0x9f, 0x99, 0xc6,
	 0x6a, 0x8c, 0xcc, 0x5c, 0xb8, 0x83, 0xfa, 0x86,
	 0x56, 0x5e, 0x91, 0x17, 0x0b, 0x5f, 0x54, 0xa8}
};
#elif defined(__riscv) && (__riscv_xlen == 32)
static u8 expected_pcrs_per_arch[][TPM2_SHA256_DIGEST_SIZE] = {
	/* PCR[4] */
	{0x64, 0xe9, 0x25, 0xb3, 0xd8, 0x33, 0xb3, 0x1b,
	 0x74, 0x0c, 0x81, 0x45, 0xef, 0x61, 0xf1, 0x87,
	 0xef, 0x65, 0x67, 0x28, 0x1a, 0x54, 0x97, 0xb2,
	 0xd3, 0x62, 0x00, 0xe7, 0xb6, 0x7a, 0xd5, 0x8e},
	/* PCR[6] */
	{0x82, 0xab, 0xc5, 0x6a, 0xbf, 0x08, 0x43, 0x3f,
	 0x85, 0xbd, 0x8f, 0x8e, 0x23, 0x62, 0x48, 0x4a,
	 0x44, 0x53, 0xf0, 0xae, 0x8d, 0x4c, 0xda, 0x04,
	 0x89, 0x9c, 0x0b, 0x81, 0x3a, 0x53, 0xf3, 0xac}
};
#elif defined(__riscv) && (__riscv_xlen == 64)
static u8 expected_pcrs_per_arch[][TPM2_SHA256_DIGEST_SIZE] = {
	/* PCR[4] */
	{0x9b, 0x5f, 0x10, 0x24, 0x28, 0x5d, 0x7d, 0x1f,
	 0x9f, 0xee, 0xe9, 0x90, 0xf1, 0x7a, 0x03, 0xb1,
	 0x68, 0x7b, 0x28, 0x45, 0x98, 0x5e, 0xf5, 0x5e,
	 0xc1, 0x22, 0x61, 0x8c, 0x2f, 0xb5, 0xbf, 0x80},
	/* PCR[6] */
	{0x6d, 0x16, 0x17, 0xf4, 0x9a, 0xa8, 0x49, 0xc2,
	 0xf4, 0x9c, 0x35, 0x30, 0x0c, 0xde, 0x65, 0xdb,
	 0xd3, 0x37, 0x9c, 0xe2, 0x9f, 0x14, 0x81, 0x74,
	 0xc3, 0x94, 0x8a, 0x9e, 0x26, 0xbf, 0xfb, 0xb2}
};
#endif

struct boot_variable {
	u16 name[16];
	u8 *buf;
	efi_uintn_t size;
	u32 attr;
	const u8 *test_data;
	efi_uintn_t test_data_size;
};

static struct boot_variable boot_variable_test[] = {
	{u"BootOrder",		NULL, 0, DEFAULT_ATTR, boot_order, sizeof(boot_order)},
	{BOOT_NAME_1000,	NULL, 0, DEFAULT_ATTR, boot_1000, sizeof(boot_1000)},
	{BOOT_NAME_1001,	NULL, 0, DEFAULT_ATTR, boot_1001, sizeof(boot_1001)},
	{BOOT_NAME_1002,	NULL, 0, DEFAULT_ATTR, boot_1002, sizeof(boot_1002)},
};

/*
 * efi_status_t decompress() - Decompress the disk image.
 *
 * @image	decompressed disk image
 * Return:	status code
 */
static efi_status_t decompress(u8 **image)
{
	u8 *buf;
	size_t i;
	size_t addr;
	size_t len;
	efi_status_t ret;

	ret = boottime->allocate_pool(EFI_LOADER_DATA, img.length,
				      (void **)&buf);
	if (ret != EFI_SUCCESS) {
		efi_st_error("Out of memory\n");
		return ret;
	}
	boottime->set_mem(buf, img.length, 0);

	for (i = 0; ; ++i) {
		if (!img.lines[i].line)
			break;
		addr = img.lines[i].addr;
		len = COMPRESSED_DISK_IMAGE_BLOCK_SIZE;
		if (addr + len > img.length)
			len = img.length - addr;
		boottime->copy_mem(buf + addr, img.lines[i].line, len);
	}
	*image = buf;
	return ret;
}

/*
 * efi_status_t setup_boot_variable() - configure dummy boot variables
 *
 * Preexisting variable values are saved and will be restored by
 * calling restore_boot_variable().
 *
 * Return:	status code
 */
static efi_status_t setup_boot_variable(void)
{
	efi_status_t ret;
	u32 i;
	efi_uintn_t size;

	for (i = 0; i < ARRAY_SIZE(boot_variable_test); i++) {
		size = 0;
		ret = runtime->get_variable(boot_variable_test[i].name,
					    &efi_global_variable_guid,
					    &boot_variable_test[i].attr,
					    &size,
					    NULL);
		if (ret == EFI_BUFFER_TOO_SMALL) {
			/* Variable exists, save the current value */
			boot_variable_test[i].size = size;
			ret = boottime->allocate_pool(EFI_LOADER_DATA,
						      boot_variable_test[i].size,
						      (void **)&boot_variable_test[i].buf);
			if (ret != EFI_SUCCESS) {
				efi_st_error("Failed to allocate buffer for boot variable\n");
				return ret;
			}
			ret = runtime->get_variable(boot_variable_test[i].name,
						    &efi_global_variable_guid,
						    &boot_variable_test[i].attr,
						    &boot_variable_test[i].size,
						    boot_variable_test[i].buf);
			if (ret != EFI_SUCCESS) {
				efi_st_error("Failed to get current boot variable\n");
				return ret;
			}
		}

		/* set boot variable for the measurement test */
		ret = runtime->set_variable(boot_variable_test[i].name,
					    &efi_global_variable_guid,
					    boot_variable_test[i].attr,
					    boot_variable_test[i].test_data_size,
					    boot_variable_test[i].test_data);
		if (ret != EFI_SUCCESS) {
			efi_st_error("Failed to set test boot variable(%d)n", i);
			return ret;
		}
	}

	return 0;
}

/*
 * efi_status_t restore_boot_variable() - restore original values
 *
 * Restore the variable values saved in setup_boot_variable().
 *
 * Return:	status code
 */
static efi_status_t restore_boot_variable(void)
{
	int i;
	efi_status_t ret;

	for (i = 0; i < ARRAY_SIZE(boot_variable_test); i++) {
		if (boot_variable_test[i].buf) {
			ret = runtime->set_variable(boot_variable_test[i].name,
						    &efi_global_variable_guid,
						    boot_variable_test[i].attr,
						    boot_variable_test[i].size,
						    boot_variable_test[i].buf);
			if (ret != EFI_SUCCESS) {
				efi_st_error("Failed to restore boot variable\n");
				return ret;
			}
			ret = boottime->free_pool(boot_variable_test[i].buf);
			if (ret != EFI_SUCCESS) {
				efi_st_error("Failed to free boot variable\n");
				return ret;
			}
		} else {
			/* delete the variable used only for testing */
			ret = runtime->set_variable(boot_variable_test[i].name,
						    &efi_global_variable_guid,
						    0, 0, NULL);
			if (ret != EFI_SUCCESS) {
				efi_st_error("Failed to delete boot variable\n");
				return ret;
			}
		}
	}

	return EFI_SUCCESS;
}

/**
 * void *find_smbios_table() - Find smbios table
 *
 * @systable	system table
 * Return:	status code
 */
static void *find_smbios_table(const struct efi_system_table *systable)
{
	u32 i;

	for (i = 0; i < systable->nr_tables; i++) {
		if (!guidcmp(&smbios3_guid, &systable->tables[i].guid))
			return systable->tables[i].table;
	}

	return NULL;
}

/**
 * efi_status_t setup_smbios_table() - Prepare the dummy SMBIOS table
 *
 * @systable	system table
 * Return:	status code
 */
static efi_status_t setup_smbios_table(const struct efi_system_table *systable)
{
	struct smbios3_entry *se;
	efi_status_t ret;
	/* Map within the low 32 bits, to allow for 32bit SMBIOS tables */
	void *dmi;

	if (sizeof(smbios3_table_test) > EFI_PAGE_SIZE)
		return EFI_OUT_OF_RESOURCES;

	orig_smbios_table = find_smbios_table(systable);

	/* Reserve 4kiB page for SMBIOS */
	ret = boottime->allocate_pages(EFI_ALLOCATE_MAX_ADDRESS,
				 EFI_RUNTIME_SERVICES_DATA, 1, &dmi_addr);

	if (ret != EFI_SUCCESS) {
		/* Could not find space in lowmem, use highmem instead */
		ret = boottime->allocate_pages(EFI_ALLOCATE_ANY_PAGES,
					 EFI_RUNTIME_SERVICES_DATA, 1,
					 &dmi_addr);

		if (ret != EFI_SUCCESS)
			return ret;
	}

	dmi = (void *)(uintptr_t)dmi_addr;
	se = dmi;
	boottime->copy_mem(se, smbios3_table_test, sizeof(smbios3_table_test));

	/* update smbios table start address */
	se->struct_table_address = (uintptr_t)((u8 *)dmi + SMBIOS3_ENTRY_HEADER_SIZE);

	se->checksum = table_compute_checksum(se, sizeof(struct smbios3_entry));

	/* Install SMBIOS information as configuration table */
	ret = boottime->install_configuration_table(&smbios3_guid, dmi);
	if (ret != EFI_SUCCESS) {
		efi_st_error("Cannot install SMBIOS table\n");
		boottime->free_pages(dmi_addr, 1);
	}

	return ret;
}

/**
 * efi_st_tcg2_setup() - setup test
 *
 * @handle:	handle of the loaded image
 * @systable:	system table
 * Return:	status code
 */
static int efi_st_tcg2_setup(const efi_handle_t img_handle,
			     const struct efi_system_table *systable)
{
	efi_status_t ret;
	struct uefi_image_load_event image_load_event;

	image_handle = img_handle;
	boottime = systable->boottime;
	runtime = systable->runtime;

	/* Load the application image into memory */
	decompress(&image);

	ret = boottime->allocate_pool(EFI_LOADER_DATA,
				      sizeof(struct efi_tcg2_event) +
				      sizeof(struct uefi_image_load_event),
				      (void **)&efi_tcg2_event);
	if (ret != EFI_SUCCESS) {
		efi_st_error("Out of memory\n");
		return EFI_ST_FAILURE;
	}

	efi_tcg2_event->size = sizeof(struct efi_tcg2_event) +
			       sizeof(struct uefi_image_load_event);
	efi_tcg2_event->header.header_size = sizeof(struct efi_tcg2_event_header);
	efi_tcg2_event->header.header_version = 1;
	efi_tcg2_event->header.pcr_index = 6;
	efi_tcg2_event->header.event_type = EV_EFI_RUNTIME_SERVICES_DRIVER;
	image_load_event.image_location_in_memory = 0x12345678;
	image_load_event.image_length_in_memory = 0x300000;
	image_load_event.image_link_time_address = 0x87654321;
	image_load_event.length_of_device_path = 0;
	boottime->copy_mem(efi_tcg2_event->event, &image_load_event,
			   sizeof(struct uefi_image_load_event));

	ret = setup_boot_variable();
	if (ret != EFI_SUCCESS)
		return EFI_ST_FAILURE;

	ret = setup_smbios_table(systable);
	if (ret != EFI_SUCCESS)
		return EFI_ST_FAILURE;

	ret = boottime->allocate_pool(EFI_LOADER_DATA,
				      (EFI_TCG2_MAX_PCR_INDEX + 1) *
				      TPM2_SHA256_DIGEST_SIZE,
				      (void **)&pcrs);
	if (ret != EFI_SUCCESS) {
		efi_st_error("Out of memory\n");
		return EFI_ST_FAILURE;
	}

	boottime->set_mem(pcrs, (EFI_TCG2_MAX_PCR_INDEX + 1) * TPM2_SHA256_DIGEST_SIZE, 0);

	/* setup expected PCRs per architecture */
	boottime->copy_mem(&expected_pcrs[4], &expected_pcrs_per_arch[0], TPM2_SHA256_DIGEST_SIZE);
	boottime->copy_mem(&expected_pcrs[6], &expected_pcrs_per_arch[1], TPM2_SHA256_DIGEST_SIZE);

	return EFI_ST_SUCCESS;
}

/**
 * efi_status_t get_manufacturer_id() - Get manufacturer_id through submit_command API
 *
 * @tcg2		tcg2 protocol
 * @manufacturer_id	pointer to the manufacturer_id
 * Return:		status code
 */
static efi_status_t get_manufacturer_id(struct efi_tcg2_protocol *tcg2, u32 *manufacturer_id)
{
	efi_status_t ret;
	u8 cmd[TPM2_CMD_BUF_SIZE] = {
		tpm_u16(TPM2_ST_NO_SESSIONS),		/* TAG */
		tpm_u32(22),				/* Length */
		tpm_u32(TPM2_CC_GET_CAPABILITY),	/* Command code */

		tpm_u32(TPM2_CAP_TPM_PROPERTIES),	/* Capability */
		tpm_u32(TPM2_PT_MANUFACTURER),		/* Property */
		tpm_u32(1),				/* Property count */
	};
	u8 resp[TPM2_CMD_BUF_SIZE];
	unsigned int value_off;

	ret = tcg2->submit_command(tcg2, 22, cmd,
				   TPM2_CMD_BUF_SIZE, resp);
	if (ret != EFI_SUCCESS)
		return ret;

	/*
	 * In the response buffer, the properties are located after the:
	 * tag (u16), response size (u32), response code (u32),
	 * YES/NO flag (u8), TPM_CAP (u32).
	 * The value is located after count (u32), property (u32).
	 */
	value_off = sizeof(u16) + sizeof(u32) + sizeof(u32) +
			 sizeof(u8) + sizeof(u32) + sizeof(u32) + sizeof(u32);
	*manufacturer_id = get_unaligned_be32(&resp[value_off]);

	return ret;
}

/**
 * efi_status_t get_manufacturer_id_buffer_small() - call submit_command with small resp buffer
 *
 * @tcg2		tcg2 protocol
 * @manufacturer_id	pointer to the manufacturer_id
 * Return:		status code
 */
static efi_status_t get_manufacturer_id_buffer_small(struct efi_tcg2_protocol *tcg2)
{
	efi_status_t ret;
	u8 cmd[TPM2_CMD_BUF_SIZE] = {
		tpm_u16(TPM2_ST_NO_SESSIONS),		/* TAG */
		tpm_u32(22),				/* Length */
		tpm_u32(TPM2_CC_GET_CAPABILITY),	/* Command code */

		tpm_u32(TPM2_CAP_TPM_PROPERTIES),	/* Capability */
		tpm_u32(TPM2_PT_MANUFACTURER),		/* Property */
		tpm_u32(1),				/* Property count */
	};
	u8 resp[1]; /* set smaller buffer than expected */

	ret = tcg2->submit_command(tcg2, 22, cmd, 1, resp);

	return ret;
}

/**
 * efi_status_t read_pcr() - Read the PCR from the TPM device
 *
 * @tcg2	tcg2 protocol
 * @idx		pcr index to read
 * Return:	status code
 */
static efi_status_t read_pcr(struct efi_tcg2_protocol *tcg2, u32 idx)
{
	efi_status_t ret;
	u32 cmd_len = 17 + IDX_ARRAY_SZ;
	u8 cmd[TPM2_CMD_BUF_SIZE] = {
		tpm_u16(TPM2_ST_NO_SESSIONS),	/* TAG */
		tpm_u32(cmd_len),		/* Length */
		tpm_u32(TPM2_CC_PCR_READ),	/* Command code */
		/* TPML_PCR_SELECTION */
		tpm_u32(1),			/* Number of selections */
		tpm_u16(TPM2_ALG_SHA256),	/* Algorithm of the hash */
		IDX_ARRAY_SZ,			/* Array size for selection */
		/* bitmap(idx),			   Selected PCR bitmap */
	};
	u8 resp[TPM2_CMD_BUF_SIZE];
	u32 pcr_sel_idx = idx / 8;
	u8 pcr_sel_bit = BIT(idx % 8);

	cmd[17 + pcr_sel_idx] = pcr_sel_bit;
	ret = tcg2->submit_command(tcg2, cmd_len, cmd,
				   TPM2_CMD_BUF_SIZE, resp);
	if (ret != EFI_SUCCESS) {
		efi_st_error("tcg2->submit_command fail to read PCR\n");
		return ret;
	}

	boottime->copy_mem(pcrs[idx], &resp[TPM2_PCR_READ_HEADER_SIZE],
			   TPM2_SHA256_DIGEST_SIZE);

	return ret;
}

/**
 * int validate_pcrs() - Compare the expected and actual pcrs
 *
 * Return:	status code
 */
static int validate_pcrs(void)
{
	u32 i;

	/*
	 *  - Skip PCR[0] validation. PCR[0] contains U-Boot version measurement
	 *    it contains the commit hash, so the measurement varies every build
	 *    with different commit hash.
	 *  - Skip PCR[7] validation. PCR[7] contains UEFI Secure Boot variables
	 *    measurement. These variables can not be updated through efi_selftest and
	 *    vary depending on the platform.
	 *  - Skip PCR[17..22] validation, they are not used in TCG PC Client
	 *    Platform Firmware Profile Specification
	 */
	for (i = 1; i < (EFI_TCG2_MAX_PCR_INDEX + 1); i++) {
		if (i == 7 || (i > 16 && i < 23))
			continue; /* skip validation */

		if (memcmp(pcrs[i], expected_pcrs[i], TPM2_SHA256_DIGEST_SIZE)) {
			efi_st_error("PCR[%d] is not the expected value\n", i);
			return EFI_ST_FAILURE;
		}
	}

	return EFI_ST_SUCCESS;
}

/**
 * efi_st_tcg2_execute() - execute test
 *
 * Call EFI_TCG2_PROTOCOL services and check the
 * Measured Boot behavior.
 *
 * Return:	status code
 */
static int efi_st_tcg2_execute(void)
{
	struct efi_tcg2_protocol *tcg2;
	struct efi_tcg2_boot_service_capability capability;
	efi_status_t ret;
	u32 active_pcr_banks;
	u64 eventlog, eventlog_last_entry;
	bool eventlog_truncated;
	efi_handle_t handle;
	efi_uintn_t exit_data_size = 0;
	u16 *exit_data = NULL;
	u32 i;
	u32 manufacturer_id;

	ret = boottime->locate_protocol(&guid_tcg2, NULL, (void **)&tcg2);
	if (ret != EFI_SUCCESS) {
		efi_st_error("TCG2 protocol is not available.\n");
		return EFI_ST_FAILURE;
	}

	/* EFI_TCG2_PROTOCOL.GetCapability test */
	capability.size = sizeof(struct efi_tcg2_boot_service_capability) - 1;
	ret = tcg2->get_capability(tcg2, &capability);
	if (ret != EFI_BUFFER_TOO_SMALL) {
		efi_st_error("tcg2->get_capability on small buffer failed\n");
		return EFI_ST_FAILURE;
	}
	capability.size = sizeof(struct efi_tcg2_boot_service_capability);
	ret = tcg2->get_capability(tcg2, &capability);
	if (ret != EFI_SUCCESS) {
		efi_st_error("tcg2->get_capability failed\n");
		return EFI_ST_FAILURE;
	}
	if (!capability.tpm_present_flag) {
		efi_st_error("TPM not present\n");
		return EFI_ST_FAILURE;
	}
	efi_st_printf("TPM supports 0x%.8x event logs\n",
		      capability.supported_event_logs);

	/* EFI_TCG2_PROTOCOL.GetActivePcrBanks test */
	ret = tcg2->get_active_pcr_banks(tcg2, &active_pcr_banks);
	if (ret != EFI_SUCCESS) {
		efi_st_error("tcg2->get_active_pcr_banks failed\n");
		return EFI_ST_FAILURE;
	}
	if (active_pcr_banks != capability.active_pcr_banks) {
		efi_st_error("tcg2->get_active_pcr_banks return wrong value\n");
		return EFI_ST_FAILURE;
	}

	/* EFI_TCG2_PROTOCOL.HashLogExtendEvent test */
	ret = tcg2->hash_log_extend_event(tcg2, EFI_TCG2_EXTEND_ONLY,
					  (uintptr_t)image,
					  img.length, efi_tcg2_event);
	if (ret != EFI_SUCCESS) {
		efi_st_error("tcg2->hash_log_extend_event(EXTEND_ONLY) failed\n");
		return EFI_ST_FAILURE;
	}

	ret = tcg2->hash_log_extend_event(tcg2, PE_COFF_IMAGE, (uintptr_t)image,
					  img.length, efi_tcg2_event);
	if (ret != EFI_SUCCESS) {
		efi_st_error("tcg2->hash_log_extend_event(PE_COFF_IMAGE) failed\n");
		return EFI_ST_FAILURE;
	}

	/* EFI_TCG2_PROTOCOL.SubmitCommand test */
	ret = get_manufacturer_id_buffer_small(tcg2);
	if (ret != EFI_OUT_OF_RESOURCES) {
		efi_st_error("get_manufacturer_id buffer too small failed\n");
		return EFI_ST_FAILURE;
	}

	ret = get_manufacturer_id(tcg2, &manufacturer_id);
	if (ret != EFI_SUCCESS) {
		efi_st_error("get_manufacturer_id failed\n");
		return EFI_ST_FAILURE;
	}
	if (capability.manufacturer_id != manufacturer_id) {
		efi_st_error("tcg2->submit_command test failed\n");
		return EFI_ST_FAILURE;
	}

	/* tcg2_measure_pe_image test */
	ret = boottime->load_image(false, image_handle, NULL, image,
				   img.length, &handle);
	if (ret != EFI_SUCCESS) {
		efi_st_error("Failed to load image\n");
		return EFI_ST_FAILURE;
	}

	/* measure ready_to_boot event(boot variables, smbios table, etc.) */
	/* TODO: add GPT measurement test */
	ret = boottime->start_image(handle, &exit_data_size, &exit_data);
	if (ret != EFI_UNSUPPORTED) {
		efi_st_error("Wrong return value from application\n");
		return EFI_ST_FAILURE;
	}
	ret = boottime->free_pool(exit_data);
	if (ret != EFI_SUCCESS) {
		efi_st_error("Failed to free exit data\n");
		return EFI_ST_FAILURE;
	}

	/* validate PCR read from the TPM device */
	for (i = 0; i < (EFI_TCG2_MAX_PCR_INDEX + 1); i++) {
		ret = read_pcr(tcg2, i);
		if (ret != EFI_SUCCESS) {
			efi_st_error("read pcr error\n");
			return EFI_ST_FAILURE;
		}
	}
	if (validate_pcrs()) {
		efi_st_error("PCR validation failed\n");
		return EFI_ST_FAILURE;
	}

	/* EFI_TCG2_PROTOCOL.GetEventLog test */
	ret = tcg2->get_eventlog(tcg2, TCG2_EVENT_LOG_FORMAT_TCG_2, &eventlog,
				 &eventlog_last_entry, &eventlog_truncated);
	if (ret != EFI_SUCCESS) {
		efi_st_error("tcg2->get_eventlog failed\n");
		return EFI_ST_FAILURE;
	}
	/* TODO: eventlog format check */

	return EFI_ST_SUCCESS;
}

/*
 * efi_st_tcg2_teardown() - Tear down unit test
 *
 * Return:	EFI_ST_SUCCESS for success
 */
static int efi_st_tcg2_teardown(void)
{
	efi_status_t r = EFI_ST_SUCCESS;

	if (image) {
		r = boottime->free_pool(image);
		if (r != EFI_SUCCESS) {
			efi_st_error("Failed to free image\n");
			return EFI_ST_FAILURE;
		}
	}
	if (efi_tcg2_event) {
		r = boottime->free_pool(efi_tcg2_event);
		if (r != EFI_SUCCESS) {
			efi_st_error("Failed to free efi_tcg2_event\n");
			return EFI_ST_FAILURE;
		}
	}
	if (pcrs) {
		r = boottime->free_pool(pcrs);
		if (r != EFI_SUCCESS) {
			efi_st_error("Failed to free pcr\n");
			return EFI_ST_FAILURE;
		}
	}

	r = restore_boot_variable();
	if (r != EFI_SUCCESS) {
		efi_st_error("Failed to restore boot variables\n");
		return EFI_ST_FAILURE;
	}

	/*
	 * Restore SMBIOS table
	 * If orig_smbios_table is NULL, calling install_configuration_table()
	 * removes dummy SMBIOS table form systab.
	 */
	r = boottime->install_configuration_table(&smbios3_guid, orig_smbios_table);
	if (r != EFI_SUCCESS) {
		efi_st_error("Failed to restore SMBOIS table\n");
		return EFI_ST_FAILURE;
	}

	if (dmi_addr) {
		r = boottime->free_pages(dmi_addr, 1);
		if (r != EFI_SUCCESS) {
			efi_st_error("Failed to free dummy smbios table\n");
			return EFI_ST_FAILURE;
		}
	}

	return r;
}

EFI_UNIT_TEST(tcg2) = {
	.name = "tcg2",
	.phase = EFI_EXECUTE_BEFORE_BOOTTIME_EXIT,
	.execute = efi_st_tcg2_execute,
	.setup = efi_st_tcg2_setup,
	.teardown = efi_st_tcg2_teardown,
	.on_request = true,
};
