/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */

// SPDX-FileCopyrightText: 2021 - 2025 Kohei Yoshida
//
// SPDX-License-Identifier: MIT

#include "test_global.hpp" // This must be the first header to be included.
#include "test_global_rtree.hpp"

using mdds::rtree;
using std::cout;
using std::endl;

void rtree_test_copy()
{
    MDDS_TEST_FUNC_SCOPE;

    using rt_type = rtree<double, double, tiny_trait_2d_forced_reinsertion>;
    using point_type = rt_type::point_type;
    rt_type::integrity_check_properties check_props;
    check_props.throw_on_first_error = false;

    struct input
    {
        point_type start;
        point_type end;
        double value;
    };

    std::vector<input> inputs = {
        {{0.0, 0.0}, {1.0, 1.0}, 1.0},    {{2.0, 2.0}, {2.1, 2.1}, 2.4},    {{100.0, 80.0}, {101.0, 85.0}, 100.0},
        {{1.0, 75.0}, {2.0, 78.0}, 65.0}, {{1.0, 80.0}, {2.0, 82.0}, 68.0}, {{1.2, 1.0}, {2.2, 1.5}, 2.1},
        {{2.2, 2.2}, {2.3, 2.4}, 3.5},    {{3.0, 3.0}, {3.3, 3.4}, 3.8},    {{4.0, 4.0}, {8.3, 12.4}, 13.8},
        {{3.0, 5.0}, {4.3, 11.4}, 13.9},
    };

    rt_type tree;
    for (const input& i : inputs)
        tree.insert({i.start, i.end}, double(i.value));

    auto copied(tree);

    tree.check_integrity(check_props);
    copied.check_integrity(check_props);

    std::string str_src = tree.export_tree(rt_type::export_tree_type::formatted_node_properties);
    std::string str_dst = tree.export_tree(rt_type::export_tree_type::formatted_node_properties);

    TEST_ASSERT(!str_src.empty() && str_src == str_dst);

    // Test the "copy via assignment" scenario too.
    auto copied_via_assign = tree;
    copied_via_assign.check_integrity(check_props);
    str_dst = copied_via_assign.export_tree(rt_type::export_tree_type::formatted_node_properties);

    TEST_ASSERT(!str_src.empty() && str_src == str_dst);
}

/**
 * Make sure the rtree works with values that are only copyable (i.e. not
 * movable).
 */
void rtree_test_only_copyable()
{
    MDDS_TEST_FUNC_SCOPE;

    using rt_type = rtree<float, only_copyable, tiny_trait_2d_forced_reinsertion>;
    using search_type = rt_type::search_type;
    using extent_type = rt_type::extent_type;
    rt_type::integrity_check_properties check_props;
    check_props.throw_on_first_error = false;

    rt_type tree;
    const rt_type& ctree = tree;
    only_copyable v(11.2);
    tree.insert({{0, 0}, {2, 5}}, v);
    v.set(12.5);
    tree.insert({9, 9}, v);

    tree.check_integrity(check_props);

    {
        // Immutable search.
        auto cres = ctree.search({1, 1}, search_type::overlap);
        TEST_ASSERT(std::distance(cres.begin(), cres.end()) == 1);
        TEST_ASSERT(cres.begin()->get() == 11.2);

        cres = ctree.search({9, 9}, search_type::overlap);
        TEST_ASSERT(std::distance(cres.cbegin(), cres.cend()) == 1);
        TEST_ASSERT(cres.cbegin()->get() == 12.5);
    }

    {
        // Mutable search
        auto res = tree.search({9, 9}, search_type::match);
        TEST_ASSERT(std::distance(res.begin(), res.end()) == 1);
        TEST_ASSERT(res.begin()->get() == 12.5);
        auto it = res.begin();
        (*it).set(34.5);

        res = tree.search({9, 9}, search_type::match);
        TEST_ASSERT(std::distance(res.begin(), res.end()) == 1);
        TEST_ASSERT(res.begin()->get() == 34.5);
    }

    {
        // Erase the only object via mutable iterator.
        TEST_ASSERT(tree.size() == 2);
        rt_type::search_results res = tree.search({{0, 0}, {100, 100}}, search_type::overlap);
        TEST_ASSERT(std::distance(res.begin(), res.end()) == 2);

        res = tree.search({9, 9}, search_type::match);
        TEST_ASSERT(std::distance(res.begin(), res.end()) == 1);
        tree.erase(res.begin());

        TEST_ASSERT(tree.size() == 1);
        res = tree.search({{0, 0}, {100, 100}}, search_type::overlap);
        TEST_ASSERT(std::distance(res.begin(), res.end()) == 1);
        auto it = res.begin();
        TEST_ASSERT(it.extent() == extent_type({{0, 0}, {2, 5}}));
        TEST_ASSERT(it->get() == 11.2);
    }
}

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
